/*
 * The noderec structure contains information about a node in the branch
 * and bound tree.
 */

#ifndef MIS_BPM_GRAPHTOOLS
#define MIS_BPM_GRAPHTOOLS

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#ifndef INCLUDED_DECLARATIONS
#define INCLUDED_DECLARATIONS
#include "declarations.h"
#endif


struct noderec {				/* node in the bandb tree */
  struct noderec *nextnode;		/* next node in the list */
  int *vals;					/* array of -1,0,1 for vertices that are unset, excluded, and
  								 * included, respectively, in the independent set */
  int *branchingvertices;		/* list of 'good' veritces to branch on */
  double bound;					/* upper bound for this node */
  double boundest;				/* estimation of the bound for SDP */
  int boundlevel;				/* level in tree at which bound was calculated */
  int level;					/* level in tree of this node */
};

void initializeGraphTools(int nodes, int edges);
/* initializeGraphTools

allocates memory for all global arrays

Input:
	vertices	the number of vertices in the full graph
	edges		the number of edges in the full graph
*/

void destroyGraphTools();

/* destroyGraphTools

deallocates memory for the global arrays 
*/

void cliqueSearch(struct noderec *currentnode, char *table, int vertices, int **neighbors, int *deg);

/* cliqueSearch

searches the unset graph stored in currentnode->vals for nodes whose neighbors are completely
connected. when found, includes the node in the independent set and excludes all neighbors. 

Input:
	currentnode		the current node in the bandb tree
	table			adjacency matrix for full graph using 'x' to denote edges, ' ' otherwise
	vertices		number of vertices in the graph
	neighbors		list of neighbors of each node
	deg				deg[i] is length of neighbors[i]
*/

void createNeighbors(char *table, int vertices, int **neighbors, int *deg);
/* createNeighbors

creates the neighbor lists from the table, 

Input:
	table			adjacency matrix for full graph using 'x' to denote edges, ' ' otherwise
	vertices		number of vertices in the graph
	neighbors		allocated array of unallocated pointers, length = vertices+1
	deg				allocated array of integers, length = vertices+1
Output:
	deg[i]			lenght of neighbors[i]
	neighbors[i] 	is allocated and holds a list of vertices adjacent to vertex i
*/
	
void heuristic(struct noderec *currentnode, int vertices, int **neighbors, int *deg, int *bestsol, int *bestval);
/* heuristic

heuristically completes the independent set in currentnode. if this new solution is a best, it is
stored in bestsol and bestval

Input:
	currentnode		the current node in the bandb tree
	vertices		number of vertices in the graph
	neighbors		list of neighbors of each node
	deg				deg[i] is length of neighbors[i]
	bestsol			current best solution known
	bestval			size of independent set in bestsol
	
Output:
	bestsol			if new best is found, this is updated
	bestval			if new best is found, this is updated
*/

int constructReducedGraph(int vertices, struct noderec *currentNode, int **neighbors, int *deg, 
	int *reducedVertices, int *nReducedEdges, int *reducedEdges, int *toReduced, double *base);
/*
Constructs the reduced graph consisiting of all unset vertices in reducedEdges

Input:
	vertices		the number of vertices in the full graph
	currentNode	the current node (containing the graph) from the BandB tree
	neighbors	neighbors[i] is a list of the neighbors of node i
	deg			deg[i] is the length of neighbors[i]
	
Output:
	returns 1 if the old X and y stored can be used for this graph, saving the pertinent 
		info in newNodeToOld and newEdgeToOld
	
	reducedVertices	set to the number of reduced vertices
	nReducedEdges	set to the number of reduced edges
	reducedEdges	stored the reduced edge set in a 2-by-nReducedEdges fortran array
	toReduced		toReduced[i]=j if the jth node in the reduced graph corresponds to the
					ith node in the full graph, 0 otherwise
	base			count of the vertices set to 1
*/

int yOldFathoms(int *edges, int nEdges, int vertices, int bestval, double base);
/*
Determines if a Z constructed from the corresponding entries in a y from a previous
bound will fathom this node. 

Inputs:
	edges	edge list for the reduced graph
	nEdges	number of reduced edges
	vertices	number of reduced vertices
	bestval the size of the largest known independent set
	base	count of all the vertices already set to 1
	
	yOld and newEdgeToOld must store the y values from the last bound and the conversion from
	the new edges to the old, respectively. 
	
Output:
	Returns 1 if the old y fathomed this node, 0 otherwise
*/

double thetabound();

#endif

