#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include "declarations.h"
#include "graphtools.h"
#include "filetools.h"

void initialize(int vertices,int edges, struct noderec **nodequeue, int **bestsol, 
		int **reducedEdges, int **toReduced, int ***neighbors, int **deg);
/* initialize

This function allocates memory for all of the arrays in main(). Input is only the number of 
vertices and edges in the graph
*/

int branchDecision(struct noderec *currentnode, int vertices, int **neighbors, int *deg);
/* branchDecision

Searches currentnode->branchingvertices for the most connected node (it will search all vertices if
branchingvertices cannot be branched on). 

Inputs:
	currentnode		the node in the abndb tree
	vertices		the number of vertices in the whole graph
	neighbors		list of neighbors of each node
	deg				deg[i] is length of neighbors[i]

Output:
	returns the node that should be branched on
*/

struct noderec* createNode(int vertices);
/* createNode
	
Allocates memory for a node in the bandb tree, does not set any values except nextnode=NULL. 
Returns a pointer to the new node. 
*/

int branch(int branchvar, int vertices, struct noderec **nodequeue, struct noderec **currentnode, 
	struct noderec **freenodes, int **neighbors, int *deg);
/* branch

Branches the current node into two new ones, one including and one excluding the vertex given in
branchvar. The includion may fail due to a conflict, in this case it returns 0. Otherwise it 
returns 1. Allocated nodes from freenodes will be used. The new nodes are added to the nodequeue

Inputs:
	branchvar		the vertex to branch on (should be set to -1)
	vertices		the number of vertices in the graph
	nodequeue		the queue of nodes to be processed
	currentnode		the current node being processed (not on the queue)
	freenodes		a list of any already allocated nodes
	neighbors		list of neighbors for each node
	deg				deg[i] is length of neighbors[i]
	
Output:
	returns 0 if the 1-branch fails, 1 otherwise
*/


int main(argc,argv)
	int argc;
	char *argv[];
{
 
	int bestval;			/* Best known value */
	int *bestsol=NULL;		/* Best known solution */
	int vertices,edges;		/* No. of vertices and edges */
	char *table;			/* adjacency table using ' ' and 'x' */
	double newbound;		/* storage for new bound */
	int branchvar;			/* storage for the next node to branch on */
	int i,count;

	/*
	* Storage for the branch and bound tree.
	*/

	struct noderec *freenodes=NULL;		/* list of nodes that are allocated but unused */
	struct noderec *nodePtr=NULL;		/* storage for creating a new node */
	struct noderec *nodequeue=NULL;		/* queue of nodes to be processed */
	struct noderec *currentnode=NULL;	/* node currently being processed */	
	
	int nodesCreated=0;		/* No. of nodes created*/
	int nodesFathomed=0;	/* No. of nodes that have been fathomed.*/
	int nodesProcessed=0;	/* No. of noes processed */
	
	/*
	* Lists of neighbors.
	*/
	
	int **neighbors=NULL;
	int *deg=NULL;			/* deg[i] is the length of neighbors[i] */
	
	/*
	* Variables related to the current solution.
	*/
	
	double currentbound;	
	double bestbound;
	int higestLevelNode;

	/*
	* variables for the reduced graph
	*/
	
	int *toReduced=NULL;			/* from node number to reduced node number in subgraph */
	int *reducedEdges=NULL;			/* edge list for the reduced graph */
	int reducedVertices, nReducedEdges, oldHelps;
	double base;					/* count of vertices already set in the independent graph at a node */
	
	/* 
	 * read the problem given the command line arguments, initialize memory, create neighbor lists
	 */
	
	table = readProblem(argc, argv, &vertices, &edges, &bestval);
	
	initializeGraphTools(vertices, edges);
	
	initialize(vertices, edges, &nodequeue, &bestsol, &reducedEdges, &toReduced, &neighbors, &deg);
	nodesCreated = 1;
	 
	createNeighbors(table, vertices, neighbors, deg);
	
	readFixed(argc, argv, nodequeue->vals, deg, neighbors);
	
	/*
	 * The main loop.
	 */
	
	while (nodequeue != NULL)
	{
		/*
		 * print some useful information every 100 nodes processed
		 */

#ifndef SILENT	
		if ((((nodesProcessed+1) % 100) == 0))
		{
			bestbound=0;
			nodePtr=nodequeue;
			count=0;
			higestLevelNode=vertices;
			while (nodePtr != NULL)
			{
				count++;
				
				if (nodePtr->bound >bestbound)
					bestbound=nodePtr->bound;
					
				if (nodePtr->level < higestLevelNode)
					higestLevelNode=nodePtr->level;
				
				nodePtr=nodePtr->nextnode;
			};
			
			printf("Processed %d total nodes, queue %d nodes, bound %f, highest node %d\n",
			nodesProcessed,count,bestbound,higestLevelNode);
			printf("Current level %d\n",currentnode->level);
			fflush(stdout);
		};
#endif
	
		/*
		 * Get a node off of the list.
		 */
		
		currentnode=nodequeue;
		nodequeue=nodequeue->nextnode;
		nodesProcessed++;
		
#ifdef VERBOSE
		if (currentnode->level <= 400)
		{
			printf("Processing a level %d node \n",currentnode->level);
			for (i=1; i<=vertices; i++)
			{
				if (currentnode->vals[i]!=-1)
				printf("%d=%d ",i,currentnode->vals[i]);
			};
			printf("\n");

			fflush(stdout);
		};
#endif




		/*
		 * Search for cliques, adjust the graph appropriately
		 */
		 
		cliqueSearch(currentnode, table, vertices, neighbors, deg);




		/*
		 * count the number of vertices set to 1 or -1, this is a very simple bound 
		 */
		
		currentbound=0.0;
		for (i=1; i<=vertices; i++)
			if ( (currentnode->vals[i]==1) || (currentnode->vals[i]==-1) )
				currentbound=currentbound+1.0;
		
		if (currentnode->bound > currentbound)
		{
			currentnode->bound=currentbound;
			currentnode->boundest=currentbound;
			currentnode->boundlevel=currentnode->level;
#ifdef VERBOSE
			printf("New best bound from unset node count: %f\n",currentbound);
			fflush(stdout);
#endif
		};
		
		/*
		 * If bound is smaller than bestval+1, then fathom.
		 */
		
		if (currentnode->bound < bestval+0.999)
		{
			currentnode->nextnode=freenodes;
			freenodes=currentnode;
			nodesFathomed++;

#ifdef VERBOSE
			printf("Fathoming node by inherited bound or unset node count\n"); 
#endif
			
			continue;
		};
		
		

		
		/*
		 * Construct the reduced graph, storing it in graphEdges
		 * oldHelps will be 1 if the new reduced graph is a subgraph of the last bounded graph
		 */
		
		oldHelps = constructReducedGraph(vertices,currentnode,neighbors,deg,&reducedVertices,
			&nReducedEdges,reducedEdges,toReduced,&base);
		/*
		 * if the reduced graph is nonexistent after the clique search then we can update bestval
		 * if necessary and fathom this node
		 */
		
		if( nReducedEdges == 0 ) 
		{
			if(base + reducedVertices > bestval)
			{
				bestval = base + reducedVertices;
				printf("New best value of %d from maximal independent set\n", bestval);
			}
			
			currentnode->nextnode=freenodes;
			freenodes=currentnode;
			nodesFathomed++;

#ifdef VERBOSE
			printf("Fathoming node with maximal independent set\n"); 
#endif
			
			continue;
		}
			

			
		
		/*
		 * if this graph is a subgraph of the last bounded graph
		 * check if the old y values will fathom this node.
		 */
#ifndef NOOLDY		
		if( (oldHelps == 1) && 
			(yOldFathoms(reducedEdges,nReducedEdges,reducedVertices,bestval,base) == 1 ) )
		{
			
			currentnode->nextnode=freenodes;
			freenodes=currentnode;
			nodesFathomed++;				
#ifdef VERBOSE
			printf("Fathoming node unisng old y values\n"); 
#endif			
			continue;				
		}
#endif
		

		
		
		/*
		 * If conditions merit, than compute an upper bound by SDP.
		 */
		
		if( (currentnode->boundest < bestval+0.999 ) || 
			( (currentnode->level-currentnode->boundlevel)>=5 ) ||
			(currentnode->level==1) ) 
		{
#ifdef NOOLDX
			oldHelps = 0;
#endif
			
			newbound = thetaBound(reducedEdges,nReducedEdges,reducedVertices,toReduced,bestval,base,
				vertices,currentnode,oldHelps);
			
			if (newbound < currentnode->bound) 
			{
				
#ifdef VERBOSE
				printf("New best bound from SDP, was %f, now %f\n",currentnode->bound,newbound);
				fflush(stdout);
#endif
				
				currentnode->boundlevel = currentnode->level;
				currentnode->bound = newbound;
				currentnode->boundest = newbound;
			} 
			else 
			{
#ifdef VERBOSE
				printf("Not an improved bound from SDP%f\n",newbound);
#endif
			}
		}
		
		/* 
		 * exit after the first bound if -1 is given as the best known solution size 
		 */
		
		if(bestval == -1)
			exit(1);
		
		/* 
		 * if we've fathomed the node
		 */
		
		if (currentnode->bound < bestval+0.999) 
		{
			
			currentnode->nextnode=freenodes;
			freenodes=currentnode;
			nodesFathomed++;
			
#ifdef VERBOSE
			printf("Fathoming node by SDP bound\n"); 
#endif
			
			continue;
		};
		
	
	
		
		/*
		 * Heuristically complete this solution.
		 */
		
		heuristic(currentnode, vertices, neighbors, deg, bestsol, &bestval);
		

		
		
		/*
		 * We need to branch. First, pick the branching variable
		 */
		
		branchvar = branchDecision(currentnode, vertices, neighbors, deg);
		
		/*
		 * Check to make sure that we got something.
		 */

		if (branchvar==0)
		{
			/*
			 * We got to the bottom. Fathom this node.
			 */
			
			currentnode->nextnode=freenodes;
			freenodes=currentnode;
			nodesFathomed++;

#ifdef VERBOSE
			printf("Fathoming node at bottom of the tree\n"); 
#endif

			continue;
		}
		else
		{
			nodesCreated++;
			if( branch(branchvar, vertices, &nodequeue, &currentnode, &freenodes, neighbors, deg) == 1)
				nodesCreated++;
			else
				nodesFathomed++;
		};




	}; /* end main loop */
	
	printf("Processed %d total nodes, queue %d nodes\n",
	nodesProcessed,nodesCreated-nodesProcessed);
	
	/*
	* We're done with the B and B.
	*/
	
	destroyGraphTools();
	free(reducedEdges);
	free(toReduced);
	free(neighbors);
	free(deg);
	
	while(freenodes !=NULL)
	{
		nodePtr = freenodes;
		freenodes = freenodes->nextnode;
		free(nodePtr);
	}
	
	printf("Best solution value %d\n",bestval);

	for (i=1; i<=vertices; i++)
	{
		if (bestsol[i]==1)
		printf(" %d ",i);
	};

	printf("\n");
	exit(0);
	
}

void initialize(vertices, edges, nodequeue, bestsol, reducedEdges, toReduced, neighbors, deg)
	int vertices;
	int edges;
	struct noderec **nodequeue;
	int **bestsol;
	int **reducedEdges;
	int **toReduced;
	int ***neighbors;
	int **deg;
{
	int i;

	/*
	 * Allocate the first node.
	 */
	
	*nodequeue=createNode(vertices);
	
	(*nodequeue)->bound=vertices;
	(*nodequeue)->boundest=vertices;
	(*nodequeue)->boundlevel=1;
	(*nodequeue)->level=1;

	for (i=1; i<=vertices; i++)
		(*nodequeue)->vals[i]=-1;
	
	/* 
	 * allocate storage for the best known solution
	 */
	
	(*bestsol)=(int *)malloc((vertices+1)*sizeof(int));
	if ((*bestsol)==NULL)
	{
		printf("Storage allocation failed!\n");
		exit(1);
	};
	
	for (i=1; i<=vertices; i++)
		(*bestsol)[i]=-1;


	/*
	 * allocate space for the conversion to the reduced problem
	 */

	(*reducedEdges) = (int *)malloc(2*edges*sizeof(double));
	(*toReduced) = (int *)malloc((vertices+1)*sizeof(int));
	
	if(((*reducedEdges) == NULL) || ((*toReduced) == NULL))
	{
		printf("Storage allocation failed!\n");
		exit(1);
	}
	
	/*
	 * Allocate space for neighbor lists
	 */
	 (*neighbors)=(int **)malloc((vertices+1)*sizeof(int *));
	
	(*deg)=(int *)malloc((vertices+1)*sizeof(int));
	if (((*deg)==NULL) ||((*neighbors)==NULL))
	{
		printf("Storage allocation failed.\n");
		exit(1);
	};
}

int branchDecision(currentnode, vertices, neighbors, deg)
	struct noderec *currentnode;
	int vertices;
	int **neighbors;
	int *deg;
{
	int branchvardeg=0;
	int branchvar=0;
	int i,j,k,count;

	for (i=1; ((i<=30) && (i<=vertices)); i++)
	{
		j=currentnode->branchingvertices[i];
		if (currentnode->vals[j]==-1)
		{
			count=0;
			for (k=1; k<=deg[j]; k++)
				if (currentnode->vals[neighbors[j][k]]==-1)
					count++;
			
			if (count > branchvardeg)
			{
				branchvardeg=count;
				branchvar=j;
			};
		};
	};
	
	if (branchvar==0)
	{
		/*
		 * Try all other variables.
		 */
		for (i=1; i<=vertices; i++)
		{
			if (currentnode->vals[i]==-1)
			{
				count=0;
				
				for (k=1; k<=deg[i]; k++)
					if (currentnode->vals[neighbors[i][k]]==-1)
						count++;
						
				if (count > branchvardeg)
				{
					branchvardeg=count;
					branchvar=i;
				};
			};
		};
	};
	
	return branchvar;
}

struct noderec * createNode(vertices)
	int vertices;
{
	struct noderec *newnode;
	
	newnode=(struct noderec *)malloc(sizeof(struct noderec));
	
	if (newnode==NULL)
	{
		printf("Storage allocation failed.\n");
		exit(1);
	};

	newnode->nextnode=NULL;	
	newnode->branchingvertices=(int *)malloc(32*sizeof(int));
	newnode->vals=(int *)malloc((vertices+1)*sizeof(int));
	
	if ((newnode->vals == NULL) || (newnode->branchingvertices == NULL))
	{
		printf("Storage allocation failed.\n");
		exit(1);
	};
	
	return newnode;
}


int branch(branchvar, vertices, nodequeue, currentnode, freenodes, neighbors, deg)
	int branchvar;
	int vertices;
	struct noderec **nodequeue;
	struct noderec **currentnode;
	struct noderec **freenodes;
	int **neighbors;
	int *deg;
{
	struct noderec *nodePtr;
	int i,j,k;

	/* 
	 * create a new node if necessary 
	 */
	 
	if ((*freenodes)==NULL)
		(*freenodes) = createNode(vertices);
	
	nodePtr=(*freenodes);
	(*freenodes)=(*freenodes)->nextnode;
	
	/*
	 * Set the zero branch first.
	 */
	
#ifdef VERBOSE
	printf("Branching on %d\n",branchvar);
#endif
	
	for (i=1; i<=vertices; i++)
		nodePtr->vals[i]=(*currentnode)->vals[i];
	
	for (i=1; ((i<=30) && (i<=vertices)); i++)
		nodePtr->branchingvertices[i]=(*currentnode)->branchingvertices[i];
	
	nodePtr->vals[branchvar]=0;
	nodePtr->boundlevel=(*currentnode)->boundlevel;
	nodePtr->bound=(*currentnode)->bound;
	nodePtr->boundest=(*currentnode)->boundest-0.05;
	nodePtr->level=(*currentnode)->level+1;
	
	/*
	 * put it on the stack
	 */
	
	nodePtr->nextnode=(*nodequeue);
	(*nodequeue)=nodePtr;
	
	/*
	 * Now, set the one branch.
	 */
	
	(*currentnode)->vals[branchvar]=1;
	
	/*
	 * Propagate the effects of this one.
	 */
	
	for (k=1; k<=deg[branchvar]; k++)
	{
		j=neighbors[branchvar][k];
		
		if ((*currentnode)->vals[j]==1)
		{
			/*
			 * Oops, we have a conflict. Since this node
			 * is infeasible, we fathom it.
			 */
			(*currentnode)->nextnode=(*freenodes);
			(*freenodes)=(*currentnode);
			return 0;
		}
		else
		{
			/*
			 * Fix this variable at 0.
			 */
			 
			(*currentnode)->vals[j]=0;
			(*currentnode)->boundest=(*currentnode)->boundest-0.05;
		};
		
	};
	
	(*currentnode)->level=(*currentnode)->level+1;
	
	/*
	* Put it on the stack.
	*/
	
	(*currentnode)->nextnode=(*nodequeue);
	(*nodequeue)=(*currentnode);
	
	return 1;
}
